#import "Common/ShaderLib/GLSLCompat.glsllib"
#import "Common/ShaderLib/Parallax.glsllib"
#import "Common/ShaderLib/BlinnPhongLighting.glsllib"
#import "Common/ShaderLib/Lighting.glsllib"

// fog - jayfella
#ifdef USE_FOG
    #import "Common/ShaderLib/MaterialFog.glsllib"
    varying float fog_distance;
    uniform vec4 m_FogColor;

    #ifdef FOG_LINEAR
        uniform vec2 m_LinearFog;
    #endif

    #ifdef FOG_EXP
        uniform float m_ExpFog;
    #endif

    #ifdef FOG_EXPSQ
        uniform float m_ExpSqFog;
    #endif

#endif // end fog

varying vec2 texCoord;

varying vec3 AmbientSum;
varying vec4 DiffuseSum;
varying vec3 SpecularSum;

uniform vec4 g_LightDirection;
varying vec3 vViewDir;
varying vec4 vLightDir;
varying vec3 lightVec;

#ifdef DIFFUSEMAP
    uniform sampler2D m_DiffuseMap;
#endif

#ifdef SPECULARMAP
    uniform sampler2D m_SpecularMap;
#endif

#ifdef PARALLAXMAP
    uniform sampler2D m_ParallaxMap;
#endif
#if (defined(PARALLAXMAP) || (defined(NORMALMAP_PARALLAX) && defined(NORMALMAP))) && !defined(VERTEX_LIGHTING)
    uniform float m_ParallaxHeight;
    varying vec3 vViewDirPrlx;
#endif

#ifdef NORMALMAP
    uniform sampler2D m_NormalMap;
#else
    varying vec3 vNormal;
#endif

#ifdef ALPHAMAP
    uniform sampler2D m_AlphaMap;
#endif

uniform float m_AlphaDiscardThreshold;
uniform float m_Alpha;
uniform float m_Shininess;


void main(){
    vec2 newTexCoord;

    #if (defined(PARALLAXMAP) || (defined(NORMALMAP_PARALLAX) && defined(NORMALMAP))) && !defined(VERTEX_LIGHTING)
        #ifdef STEEP_PARALLAX
            #ifdef NORMALMAP_PARALLAX
                //parallax map is stored in the alpha channel of the normal map
                newTexCoord = steepParallaxOffset(m_NormalMap, vViewDirPrlx, texCoord, m_ParallaxHeight);
            #else
                //parallax map is a texture
                newTexCoord = steepParallaxOffset(m_ParallaxMap, vViewDirPrlx, texCoord, m_ParallaxHeight);
            #endif
        #else
            #ifdef NORMALMAP_PARALLAX
                //parallax map is stored in the alpha channel of the normal map
                newTexCoord = classicParallaxOffset(m_NormalMap, vViewDirPrlx, texCoord, m_ParallaxHeight);
            #else
                //parallax map is a texture
                newTexCoord = classicParallaxOffset(m_ParallaxMap, vViewDirPrlx, texCoord, m_ParallaxHeight);
            #endif
        #endif
    #else
        newTexCoord = texCoord;
    #endif

    #ifdef DIFFUSEMAP
        vec4 diffuseColor = texture2D(m_DiffuseMap, newTexCoord);
    #else
        vec4 diffuseColor = vec4(1.0);
    #endif

    float alpha = DiffuseSum.a * diffuseColor.a;
    #ifdef ALPHAMAP
        alpha = alpha * texture2D(m_AlphaMap, newTexCoord).r;
    #endif

    #ifdef DISCARD_ALPHA
        if(alpha < m_AlphaDiscardThreshold){
            discard;
        }

        // Else force alpha to solid
        // 2021-11-24 - as I recall, this prevents the trees from fading to half
        // transparent because of min texture filtering as they get farther away
        diffuseColor = mix(diffuseColor, vec4(0.0, 0.1, 0.0, 1.0), 1.0 - alpha);
        //alpha = 1.0;
        alpha = m_Alpha;
    #endif


    // ***********************
    // Read from textures
    // ***********************
    #if defined(NORMALMAP)
        vec4 normalHeight = texture2D(m_NormalMap, newTexCoord);
        //Note the -2.0 and -1.0. We invert the green channel of the normal map,
        //as it's compliant with normal maps generated with blender.
        //see http://hub.jmonkeyengine.org/forum/topic/parallax-mapping-fundamental-bug/#post-256898
        //for more explanation.
        vec3 normal = normalize((normalHeight.xyz * vec3(2.0,-2.0,2.0) - vec3(1.0,-1.0,1.0)));

        // 2021-11-24 - I'm not sure what LATC does but I'm leaving the commented
        // out code in for reference.  Nothing triggers this define as far as I can tell.
        //#ifdef LATC
        //  normal.z = sqrt(1.0 - (normal.x * normal.x) - (normal.y * normal.y));
        //#endif
    #else
        vec3 normal = vNormal;
        normal = normalize(normal);
    #endif

    #ifdef SPECULARMAP
        vec4 specularColor = texture2D(m_SpecularMap, newTexCoord);
    #else
        vec4 specularColor = vec4(1.0);
    #endif

    vec4 lightDir = vLightDir;
    lightDir.xyz = normalize(lightDir.xyz);
    vec3 viewDir = normalize(vViewDir);
    float spotFallOff = 1.0;


    // allow use of control flow
    if(g_LightDirection.w != 0.0) {
        spotFallOff =  computeSpotFalloff(g_LightDirection, lightVec);
        if(spotFallOff <= 0.0) {
            gl_FragColor.rgb = AmbientSum * diffuseColor.rgb;
            gl_FragColor.a   = alpha;
            return;
        }
    }

    vec2 light = computeLighting(normal, viewDir, lightDir.xyz, lightDir.w * spotFallOff, m_Shininess) ;

    // Workaround, since it is not possible to modify varying variables
    vec4 SpecularSum2 = vec4(SpecularSum, 1.0);

    gl_FragColor.rgb =  AmbientSum       * diffuseColor.rgb  +
                        DiffuseSum.rgb   * diffuseColor.rgb  * vec3(light.x) +
                        SpecularSum2.rgb * specularColor.rgb * vec3(light.y);

    // add fog after the lighting because shadows will cause the fog to darken
    // which just results in the geometry looking like it's changed color
    #ifdef USE_FOG
        #ifdef FOG_LINEAR
            gl_FragColor = getFogLinear(gl_FragColor, m_FogColor, m_LinearFog.x, m_LinearFog.y, fog_distance);
        #endif
        #ifdef FOG_EXP
            gl_FragColor = getFogExp(gl_FragColor, m_FogColor, m_ExpFog, fog_distance);
        #endif
        #ifdef FOG_EXPSQ
            gl_FragColor = getFogExpSquare(gl_FragColor, m_FogColor, m_ExpSqFog, fog_distance);
        #endif
    #endif // end fog

    gl_FragColor.a = alpha;
}
