#import "Common/ShaderLib/GLSLCompat.glsllib"
#import "Common/ShaderLib/BlinnPhongLighting.glsllib"
#import "Common/ShaderLib/Lighting.glsllib"

//#define DEBUG_MASKING 1

#import "MatDefs/terrain/TerrainTiles.glsllib"
#import "MatDefs/terrain/TerrainTypes.glsllib"

// fog - jayfella
#ifdef USE_FOG
    #import "Common/ShaderLib/MaterialFog.glsllib"
    varying float fog_distance;
    uniform vec4 m_FogColor;

    #ifdef FOG_LINEAR
        uniform vec2 m_LinearFog;
    #endif

    #ifdef FOG_EXP
        uniform float m_ExpFog;
    #endif

    #ifdef FOG_EXPSQ
        uniform float m_ExpSqFog;
    #endif

#endif // end fog

varying vec3 AmbientSum;
varying vec4 DiffuseSum;
varying vec3 SpecularSum;
uniform float m_Shininess;

uniform vec4 g_LightDirection;
varying vec3 vViewDir;
varying vec4 vLightDir;
varying vec3 lightVec;

varying vec3 vNormal;

uniform float m_AlphaDiscardThreshold;

varying vec3 mPos;
varying vec3 wNormal;

#if defined(HORIZONTAL_LIMIT) && defined(VERTICAL_LIMIT)
    const float INV_RANGE = 1.0 / (VERTICAL_LIMIT - HORIZONTAL_LIMIT);
#endif

void main(){

    vec2 texCoord;

    // We lock this behind a #define because it avoids having the discard
    // compiled in when there is no view mask.  The discard keyword can change
    // how a shader is optimized even if that path is never taken... and if
    // we don't need it then no reason to bother with it.
    #if defined(HAS_VIEW_MASK) || defined(HAS_MASK_ARRAY)
        vec4 maskValue = getViewMask(mPos);
        if( maskValue.r > 0.6 ) {
            discard;
        }
    #endif

    vec2 terrainPos = modelToTerrainPos(mPos);
    texCoord = terrainPosToUv(terrainPos);

    vec2 worldData = getWorldData(terrainPos);
    float terrainType = worldData.x;
    float lightAdjust = m_LocalLightStrength * worldData.y;

    #if defined(HORIZONTAL_LIMIT) && defined(VERTICAL_LIMIT)
        vec3 worldNormal = normalize(wNormal);
        float upness = clamp(worldNormal.y, HORIZONTAL_LIMIT, VERTICAL_LIMIT);
        upness = (upness - HORIZONTAL_LIMIT) * INV_RANGE;
    #else
        vec3 worldNormal = wNormal;
        // Garbage but better than an arbitrary value
        // less garbage if we normalized the normal but then we'd be
        // doing work that the getTerrainDiffuse() call probably isn't
        // using.
        //float upness = wNormal.y;
        float upness = 1.0; // easier to see when we have it on/off... and the
                            // function can always just grab the y from world normal
                            // if it wants to.
    #endif

    vec4 diffuseColor = getTerrainDiffuse(terrainType, terrainPos, worldNormal, upness);

    #ifdef DEBUG_MASKING
        #if defined(HAS_VIEW_MASK) || defined(HAS_MASK_ARRAY)
            diffuseColor.rgb = maskValue.gba;
        #endif
    #endif

    float alpha = DiffuseSum.a * diffuseColor.a;
    #ifdef DISCARD_ALPHA
        if(alpha < m_AlphaDiscardThreshold){
            discard;
        }
    #endif

    vec3 normal = normalize(vNormal);

    vec4 specularColor = vec4(1.0);

    vec4 lightDir = vLightDir;
    lightDir.xyz = normalize(lightDir.xyz);
    vec3 viewDir = normalize(vViewDir);
    float spotFallOff = 1.0;

    // allow use of control flow
    if( g_LightDirection.w != 0.0 ) {
        spotFallOff =  computeSpotFalloff(g_LightDirection, lightVec);
        if( spotFallOff <= 0.0 ) {
            gl_FragColor.rgb = AmbientSum * diffuseColor.rgb;
            gl_FragColor.a   = alpha;
            return;
        }
    }

    vec2 light = computeLighting(normal, viewDir, lightDir.xyz, lightDir.w * spotFallOff, m_Shininess) ;

    // Workaround, since it is not possible to modify varying variables
    vec4 SpecularSum2 = vec4(SpecularSum, 1.0);

    float sunContrib = min(1.0, length(DiffuseSum.rgb) * light.x);
    float localContrib = min(1.0, 1.25 - sunContrib);

    gl_FragColor.rgb =  AmbientSum       * diffuseColor.rgb  +
                        DiffuseSum.rgb   * diffuseColor.rgb  * vec3(light.x) +
                        //DiffuseSum.rgb   * diffuseColor.rgb  * lightAdjust +
                        diffuseColor.rgb  * lightAdjust * localContrib +
                        SpecularSum2.rgb * specularColor.rgb * vec3(light.y);

    // add fog after the lighting because shadows will cause the fog to darken
    // which just results in the geometry looking like it's changed color
    #ifdef USE_FOG
        #ifdef FOG_LINEAR
            gl_FragColor = getFogLinear(gl_FragColor, m_FogColor, m_LinearFog.x, m_LinearFog.y, fog_distance);
        #endif
        #ifdef FOG_EXP
            gl_FragColor = getFogExp(gl_FragColor, m_FogColor, m_ExpFog, fog_distance);
        #endif
        #ifdef FOG_EXPSQ
            gl_FragColor = getFogExpSquare(gl_FragColor, m_FogColor, m_ExpSqFog, fog_distance);
        #endif
    #endif // end fog

    gl_FragColor.a = alpha;
}
