
println "Initializing preload-server-commands.groovy"

def timeString = { nanos ->
    def ms = nanos/1000000.0;
    if( ms < 1000 ) {
        return String.format("%.3f ms", ms);
    }
    def secs = ms / 1000;
    if( secs < 60 ) {
        return String.format("%.2f s  (total ms: %.3f)", secs, ms);
    }
    int minutes = (int)(secs / 60);
    secs = (double)secs % 60;
    return String.format("%d m %.2f s  (total ms: %.3f)", minutes, secs, ms);
}


def help = [
    "[radius] - preloads an area around the spawn.",
    "Where:",
    "  [radius] - is an optional radius in col:hi:low format.",
    "       col - indicates the number of columns to load around",
    "             the spawn point.",
    "       hi - indicates the number of hi-res tiles to load around",
    "             the spawn tile.",
    "       low - indicates the number of lo-res tiles to load around",
    "             the spawn tile.",
    "     Defaults to 3:2:7 which will preload a 7x7 area of columns,",
    "     a 5x5 area of hi-res tiles (terrain, trees, buildings), and",
    "     a 15x15 area of low-res terrain tiles."
    ];
updateAdminCommand("preload", help) { shell, args ->
    // Note: the client preloader defaults
    // to 2:1:0

    def colRadius = 3;
    def tileRadius = 2;
    def lowRadius = 7;

    def tokens = args?.split(":");
    if( tokens ) {
        colRadius = Integer.parseInt(tokens[0]);
        if( tokens.length > 1 ) {
            tileRadius = Integer.parseInt(tokens[1]);
        }
        if( tokens.length > 2 ) {
            lowRadius = Integer.parseInt(tokens[2]);
        }
    }

    def spawn = worldManager.info.spawnPoint;
    shell.echo("Loading area around:" + spawn);
    def colSize = 1 + colRadius * 2;
    def tileSize = 1 + tileRadius * 2;
    def lowSize = 1 + lowRadius * 2;
    shell.echo("    " + colSize + "x" + colSize + " columns");
    shell.echo("    " + tileSize + "x" + tileSize + " hi-res tiles");
    shell.echo("    " + lowSize + "x" + lowSize + " low-res tiles");

    long start = System.nanoTime();

    def tiles = [] as Set;
    for( int x = -tileRadius; x <= tileRadius; x++ ) {
        for( int z = -tileRadius; z <= tileRadius; z++ ) {
            TileId tileId = TileId.fromWorld(spawn.add(x * TileId.SIZE, 0, z * TileId.SIZE));
            tiles.add(tileId);
            shell.echo("Loading tile:" + tileId.getWorld(null) + " terrain...");
            world.getTerrainImage(tileId, TerrainImageType.Terrain, Resolution.High);
            shell.echo("Loading tile:" + tileId.getWorld(null) + " trees...");
            world.getTrees(tileId, Resolution.High);
            shell.echo("Loading tile:" + tileId.getWorld(null) + " far buildings...");
            world.getPointCloudLayer(tileId, Resolution.High);
        }
    }

    for( int x = -lowRadius; x <= lowRadius; x++ ) {
        for( int z = -lowRadius; z <= lowRadius; z++ ) {
            TileId tileId = TileId.fromWorld(spawn.add(x * TileId.SIZE, 0, z * TileId.SIZE));
            if( tiles.contains(tileId) ) {
                // The hi-res load would have generated the lo-res
                continue;
            }
            shell.echo("Loading tile:" + tileId.getWorld(null) + " low-res terrain...");
            world.getTerrainImage(tileId, TerrainImageType.Terrain, Resolution.Low);
        }
    }

    for( int x = -colRadius; x <= colRadius; x++ ) {
        for( int z = -colRadius; z <= colRadius; z++ ) {
            ColumnId colId = ColumnId.fromWorld(spawn.add(x * ColumnId.SIZE, 0, z * ColumnId.SIZE));

            // Loading any leaf from the column should cause the whole thing to generate
            LeafId leafId = LeafId.fromWorld(spawn.add(x * ColumnId.SIZE, 0, z * ColumnId.SIZE));
            shell.echo("Loading column:" + colId.getWorld(null));
            world.getLeaf(leafId);
        }
    }

    long end = System.nanoTime();
    shell.echo("******************************************");
    shell.echo("***** Preloaded world in " + timeString(end - start));
    shell.echo("******************************************");
}


