#import "Common/ShaderLib/GLSLCompat.glsllib"
#import "Common/ShaderLib/Instancing.glsllib"
#import "Common/ShaderLib/Lighting.glsllib"

#import "MatDefs/ViewMask.glsllib"

// fog - jayfella
#ifdef USE_FOG
    varying float fog_distance;
#endif
uniform vec3 g_CameraPosition;

uniform vec4 m_Ambient;
uniform vec4 m_Diffuse;
uniform vec4 m_Specular;
uniform float m_Shininess;

uniform vec4 g_LightColor;
uniform vec4 g_LightPosition;
uniform vec4 g_AmbientLightColor;

varying vec2 texCoord;

varying vec3 AmbientSum;
varying vec4 DiffuseSum;
varying vec3 SpecularSum;

attribute vec3 inPosition;
attribute vec2 inTexCoord;
attribute float inSize;

varying vec3 lightVec;

attribute vec4 inTangent;

#ifndef NORMALMAP
    varying vec3 vNormal;
#endif
varying vec3 vViewDir;
varying vec4 vLightDir;

#if (defined(PARALLAXMAP) || (defined(NORMALMAP_PARALLAX) && defined(NORMALMAP))) && !defined(VERTEX_LIGHTING)
    varying vec3 vViewDirPrlx;
#endif


void main(){
    vec4 modelSpacePos = vec4(inPosition, 1.0);

    // The texture coordinate defines which of the four corners this vertex
    // is.  The inSize parameter is ultimately what determines the atlas cell.
    // The texture coordinate will be used to looking up texture colors as well
    // as projecting the corner vertexes in the right direction relative to camera
    // direction.
    texCoord = inTexCoord;

    vec4 maskValue = getViewMask(modelSpacePos.xyz);
    if( maskValue.r > 0.5 ) {
        #ifndef DEBUG_VIEW_MASK
            // Make the tree render nothing.
            texCoord *= 0.0;
        #else
            // Make it stand out
            texCoord *= 1.5;
        #endif
    }

    // The world position for each of the four corners is the same... the
    // center base of the billboard.
    vec3 wPosition = (g_WorldMatrix * modelSpacePos).xyz;
    vec3 offset = wPosition - g_CameraPosition;


    // Figure out which way to project our corner
    vec3 dir = normalize(offset);
    vec3 left = normalize(cross(dir, vec3(0.0, 1.0, 0.0)));
    vec3 tipDir = normalize(vec3(dir.x, 0.0, dir.z));

    // Only project out the corners if the tree is within the clip range.
    // Else we'll let it just be a single point.
    float size = step(length(offset.xz), CLIP_DISTANCE);

    // Adjust the position according to the texture vectors
    // and the camera direction.
    wPosition += left * texCoord.x * size;
    wPosition.y += texCoord.y * size;

    // And we'll tip the tops back a little bit when looking
    // straight down or nearly so
    wPosition += -dir.y * tipDir * texCoord.y * 0.1 * size;

    // We'll make a normal that leads to sort of an artificial curve
    // shape.  With some tweaks for lighting direction to get more
    // aesthetically pleasing results.

    // We want backfacing to be 1 for completely facing away from the light
    // and 0 for completely facing it.  So we shift the -1 to 1 into the 0 to 1 realm
    vec3 lightDir = normalize(g_LightPosition.xyz);
    float backFacing = (1.0 + dot(-tipDir, lightDir)) * 0.5;

    // 2021-11-24 - These next two lines of math are trial-and-error magic that I no
    // longer remember an exact explanation for.  Probably worth reverse-engineering
    // the parts to remember what they do and document them.  Lighting is not correct
    // at the moment (too dark in direct sun) and these could be part of the reason.
    vec3 modelSpaceNorm = mix(-tipDir, vec3(0.0, max(0.25, sign(texCoord.y)), 0.0), backFacing) + left * sign(texCoord.x) * 0.5;

    // That one is really close but just slightly too dark
    // ...apparently really dark.
    modelSpaceNorm += vec3(0.0, 8.0, 0.0) * max(0.25, sign(texCoord.y));

    vec3 wvPosition = (g_ViewMatrix * vec4(wPosition, 1.0)).xyz;
    gl_Position = g_ViewProjectionMatrix * vec4(wPosition, 1.0);

    // Then fix the texture coordinates
    texCoord.x = (sign(texCoord.x) + 1.0) * 0.5;// * 0.124;
    texCoord.y = sign(texCoord.y);// * 0.249;

    // Now adjust for the current cell
    // 2021-11-24 - FIXME: support more than just a 4x4 atlas
    float cell = inSize;
    texCoord.x = mod(cell, 4.0)/4.0 + texCoord.x * 0.249;
    texCoord.y = 0.75 + texCoord.y * 0.249 - floor(cell/4.0) * 0.25;

    vec3 wvNormal  = normalize(TransformNormal(modelSpaceNorm));
    vec3 viewDir = normalize(-wvPosition);

    vec4 wvLightPos = (g_ViewMatrix * vec4(g_LightPosition.xyz,clamp(g_LightColor.w,0.0,1.0)));
    wvLightPos.w = g_LightPosition.w;
    vec4 lightColor = g_LightColor;

    #if (defined(NORMALMAP) || defined(PARALLAXMAP)) && !defined(VERTEX_LIGHTING)
        // 2021-11-24 - adding this wvTangent code calculation untested so
        // I'm leaving the original Lighting.j3md line commented out
        //vec3 wvTangent = normalize(TransformNormal(modelSpaceTan));
        vec3 wvTangent = -left;
        vec3 wvBinormal = cross(wvNormal, wvTangent);
        mat3 tbnMat = mat3(wvTangent, wvBinormal * inTangent.w,wvNormal);
    #endif

    #if defined(NORMALMAP)
        vViewDir  = -wvPosition * tbnMat;
        #if (defined(PARALLAXMAP) || (defined(NORMALMAP_PARALLAX) && defined(NORMALMAP)))
            vViewDirPrlx = vViewDir;
        #endif
        lightComputeDir(wvPosition, lightColor.w, wvLightPos, vLightDir, lightVec);
        vLightDir.xyz = (vLightDir.xyz * tbnMat).xyz;
    #else
        vNormal = wvNormal;
        vViewDir = viewDir;
        #if defined(PARALLAXMAP)
            vViewDirPrlx  =  -wvPosition * tbnMat;
        #endif
        lightComputeDir(wvPosition, lightColor.w, wvLightPos, vLightDir, lightVec);
    #endif

    #ifdef MATERIAL_COLORS
        AmbientSum  = (m_Ambient  * g_AmbientLightColor).rgb;
        DiffuseSum  =  m_Diffuse  * vec4(lightColor.rgb, 1.0);
        SpecularSum = (m_Specular * lightColor).rgb;
    #else
        // Defaults: Ambient and diffuse are white, specular is black.
        AmbientSum  = g_AmbientLightColor.rgb;
        DiffuseSum  =  vec4(lightColor.rgb, 1.0);
        SpecularSum = vec3(0.0);
    #endif

    #ifdef USE_FOG
        fog_distance = distance(g_CameraPosition, (g_WorldMatrix * modelSpacePos).xyz);
    #endif

    // For debugging the backfacing parameter
    //DiffuseSum.r = backFacing;
    //DiffuseSum.g = 0.0;
    //DiffuseSum.b = 0.0;

    #ifdef DEBUG_VIEW_MASK
        if( maskValue.r > 0.5 ) {
            DiffuseSum.r = 1.0;
            DiffuseSum.g = 0.0;
            DiffuseSum.b = 0.0;
        }
    #endif
}
