
import java.util.concurrent.TimeUnit;
import com.google.common.cache.*;


// This will eventually be some other kind of object for the
// permissions chain
defaultWorldPermissions = new ClaimPermissions(null,
                                    ClaimPermissions.BLOCK_ADD,
                                    ClaimPermissions.BLOCK_REMOVE,
                                    ClaimPermissions.OBJ_ADD,
                                    ClaimPermissions.OBJ_REMOVE,
                                    ClaimPermissions.OBJ_MOVE,
                                    ClaimPermissions.OBJ_CHANGE,
                                    ClaimPermissions.PROP_ADD,
                                    ClaimPermissions.PROP_REMOVE,
                                    ClaimPermissions.PROP_MOVE,
                                    ClaimPermissions.PROP_CHANGE);

/**
 *  Loads a list of claims that might intersect the specified zoneId.
 */
loadClaims = { long zoneId ->
    log.info("loadClaims(" + zoneId + ")");
    def grid = ClaimArea.getZoneGrid();
    def centerCell = grid.idToCell(zoneId, null);
    int radius = 1;
    int size = radius * 2 + 1;
    ComponentFilter[] filters = new ComponentFilter[size * size];

    int index = 0;
    for( int x = -radius; x <= radius; x++ ) {
        for( int z = -radius; z <= radius; z++ ) {
            long id = grid.cellToId(centerCell.x + x, centerCell.y, centerCell.z + z);
            def filter = ClaimArea.zoneFilter(id);
            filters[index++] = filter;
        }
    }

    return new ArrayList(findEntities(Filters.or(ClaimArea.class, filters), ClaimArea));
}

// Note: the claims cache doesn't seem to make a huge difference when there are just
// a few claims.  I suspect it will matter more when there are a lot more properties
// but we'll have to see.
claimsCache = CacheBuilder.newBuilder()
                .maximumSize(100)
                .build(loadClaims);

// When a new claim comes in, we have to make sure the cache is emptied.  The zones
// are really large and it's quite probably that players pounding away in a given zone
// would prevent the cache from refreshing naturally if it had just a short expiration.
// So instead, we'll leave expiration infinite but purge the cache whenever a claim
// area changes.
entityData.addEntityComponentListener(new CachePurger(claimsCache, ClaimArea.class));

/**
 *  Find any claims that might by in the vicinity of the
 *  specified location.
 */
findClaims = { pos ->
    if( pos == null ) {
        log.warn("pos is null in findClaims()", new Throwable("stack-trace"));
        return;
    }
    if( log.isTraceEnabled() ) {
        log.trace("findClaims(" + pos + ")");
    }

    long id = ClaimArea.calculateZoneId(pos as Vec3d);
    return claimsCache.get(id);
}

getIntersections = { area ->
    def center = (area.min + area.max) / 2;
    def results = [] as Set;
    findClaims(center).each { it ->
        if( area.intersects(it[ClaimArea]) ) {
            results.add(it);
        }
    }
    return results;
}

getClaim = { Activator activator, loc ->

    if( loc instanceof Vec3d ) {
        loc = loc.floor();
    }

    // Caching is tricky because we'd need to watch for claim area updates
    // and flush the cache.  We should do it... but it will wait until things
    // are working.  And it will be a proper area cache, not an activator-based cache
    //def lastLoc = activator.getProperty("lastClaimLoc")?.floor();
    ////def loc = activator.getPos()?.floor();
    //
    //if( loc == lastLoc ) {
    //    return activator.getProperty("claim");
    //}

    // Else we need to find the claim
    for( EntityId claim : findClaims(loc) ) {
        if( log.isDebugEnabled() ) {
            log.debug("Checking:" + claim + " contains(" + loc + "):" + claim[ClaimArea].contains(loc));
        }
        if( claim[ClaimArea].contains(loc) ) {
            return claim;
        }
    }

    return null;
}


getPerms = { Activator activator, loc ->

long startTime = System.nanoTime();
try {
    // If we are admin then we always have permission
    if( activator.getProperty("account")?.getProperty("permissions", List.class)?.contains("admin") ) {;
        return defaultWorldPermissions;
    }

    def claim = getClaim(activator, loc);
    if( log.isDebugEnabled() ) {
        log.debug("claim:" + claim);
    }
    if( claim == null ) {
        return defaultWorldPermissions;
    }

    if( log.isDebugEnabled() ) {
        log.debug("owner:" + claim[OwnedBy].owner + " us:" + activator);
    }
    // Else is it ours?
    if( activator.isSameEntity(claim[OwnedBy].owner) ) {
        return ClaimPermissions.createOwnerPermissions(claim);
    }

    if( log.isDebugEnabled() ) {
        log.debug("no access");
    }
    // For the moment... we really want to look for entities on our
    // person that give us permission
    return ClaimPermissions.createNoAccess(claim);
} finally {
    long endTime = System.nanoTime();
    log.info(String.format("getPerms() %.03f ms", (endTime - startTime) / 1000000.0));
}
}

findProperty = { EntityId owner ->
    def filter = Filters.fieldEquals(OwnedBy, "owner", owner);
    return findEntities(filter, OwnedBy, ClaimType);
}

grantProperty = { EntityId target, String name, ClaimType type ->

    def claim = createEntity(
            new Name(name),
            ObjectName.create("Claim Area", entityData),
            ObjectTypeInfo.create("Claim", entityData),
            new ContainedIn(target, target, 0, 0),
            type,
            new OwnedBy(target)
        );

    // It doesn't get position, area, etc. until placed
    return claim;
}


// The default property grants
on( playerEntityCreated ) { event ->
}

on( playerEntityJoined ) { event ->
    // Since we already have players without strongholds we'll
    // check for missing strongholds on join instead of relying on created

    def stronghold = findProperty(event.player).find { it[ClaimType]?.type == ClaimType.TYPE_STRONGHOLD };

    log.info("Found existing stronghold:" + stronghold);

    if( stronghold == null ) {
        def name = event.player.name;
        log.info("Creating stronghold for player:" + event.player + " name:" + name);
        grantProperty(event.player, "" + name + "'s Stronghold",
                      ClaimType.createStandard(ClaimType.TYPE_STRONGHOLD));
    } else {
        // We could uprgade it if we forgot something
    }
}

