/*
 * $Id$
 * 
 * Copyright (c) 2022, Simsilica, LLC
 * All rights reserved.
 */

MOON_DIRECT_MIN = new ColorRGBA(0.05f, 0.05f, 0.05f, 1f);  
MOON_DIRECT_MAX = new ColorRGBA(0.1f, 0.1f, 0.2f, 1f);
MOON_AMBIENT_MIN = new ColorRGBA(0.025f, 0.025f, 0.05f, 1f); 
MOON_AMBIENT_MAX = new ColorRGBA(0.05f, 0.05f, 0.1f, 1f); 

// We should be able to set these on the fly in one tween and have
// them change as parameters to another tween
moonDirectColor = new ColorRGBA(0.1f, 0.1f, 0.2f, 1f); 
moonAmbientColor = new ColorRGBA(0.05f, 0.05f, 0.1f, 1f); 

// Sets the phase and angle of the moon based on day
addSchedule("25 0:00").with {
    interpolate { t, start, end ->
        def val = start + (end - start) * t;
        //log.info("val:" + val + "  start:" + start + "  end:" + end + "  t:" + t);
        int phase = (int)Math.round(val);
        env.setMoonPhase(phase);
        
        float angle = 0.5f * ((phase - 12) / 12f);
        env.setMoonPhaseAngle(angle);
        
        // Rather than create another schedule we'll just use the calculated
        // phase here.  Any interpolator would give us slight partial results
        // over the night but we want color to be constant over a single night
        // so even in a separate tween we'd clamp to phase.
        // Max brightess is at phase 12 (center of the texture) with 0 and 24
        // both being total darkness
        double lightInterp = 1 - Math.abs(12 - phase)/12.0;
        def direct = MOON_DIRECT_MIN + (MOON_DIRECT_MAX - MOON_DIRECT_MIN) * lightInterp;
        moonDirectColor.set(direct);
        
        def ambient = MOON_AMBIENT_MIN + (MOON_AMBIENT_MAX - MOON_AMBIENT_MIN) * lightInterp;
        moonAmbientColor.set(ambient);
    }
    at("0 0:00", 0)
    at("24 0:00", 24)
    at("24 12:00", 24)
    at("24 13:01", 0)
}


// Sets the base color of the moon image based on time of day
addSchedule("24:00").with {
    interpolate { t, start, end ->
        def val = start + (end - start) * t;
        env.setMoonColor(val);
    }

    // Based on 6:08 - 19:52 day.
    at("0:00", new ColorRGBA(1f, 1f, 1f, 1f));
    at("5:00", new ColorRGBA(1f, 1f, 1f, 1f));
    at("6:30", new ColorRGBA(0.25f, 0.25f, 0.25f, 1f));
    at("19:30", new ColorRGBA(0.25f, 0.25f, 0.25f, 1f));
    at("20:00", new ColorRGBA(1f, 1f, 1f, 1f));
}

// Sets where the starts fade for the daylight portion of the sky
// based on time of day.
addSchedule("24:00").with {
    interpolate { t, start, end ->
        def val = start + (end - start) * t;
        env.setSunFade(val);
    }
 
    if( false ) {
    at("0:00", new Vector2f(500, 500))
    at("5:00", new Vector2f(500, 500))
    at("5:30", new Vector2f(450, 500))
    at("6:00", new Vector2f(400, 480))
    at("6:30", new Vector2f(0, 0))
    at("17:00", new Vector2f(0, 300)) 
    at("18:00", new Vector2f(250, 350)) 
    at("18:30", new Vector2f(350, 450)) 
    at("18:45", new Vector2f(400, 500)) 
    at("19:00", new Vector2f(500, 500))
    } else {
    // Adapted to a 6:08 - 19:52 day.  (+08 and +82)   
    at("0:00", new Vector2f(500, 500))
    at("5:00", new Vector2f(500, 500))
    at("5:15", new Vector2f(450, 500))
    at("5:38", new Vector2f(300, 480))
    at("6:08", new Vector2f(150, 280))
    at("6:38", new Vector2f(0, 0))
    at("18:52", new Vector2f(0, 300)) 
    at("19:52", new Vector2f(250, 350)) 
    at("20:22", new Vector2f(350, 450)) 
    at("20:37", new Vector2f(400, 500)) 
    at("20:52", new Vector2f(500, 500))
    }
}    

// Tweaks to the angle of the moon relative to time of day so that the
// shorter night will still allow the moon to go horizon to horizon
// when the sun rises/sets.
addSchedule("24:00").with {
    interpolate { t, start, end ->
        def both = start + (end - start) * t;
        
        int phase = env.getMoonPhase();
        
        // We offset the regular advance based on the moon phase to
        // keep it level with the horizon when at the extremes.
        float offset = (phase - 12)/12.0;
        float scale = both.y;
        //log.info("val:" + val + " scale:" + scale);         
        //val = (val * 0.1f) + offset * 0.21 * ((val - 0.5) * 2);
        //val = (val * -0.4f) + offset * 0.22 * ((val - 0.5) * 2);
        //double val = both.x * -0.4f + offset * 0.22 * scale;  
        double val = both.x + offset * 0.22 * scale;  
        
        env.setMoonAdvance((float)val);
    }    

    // Based on 6:08 - 19:52 day.
    at("0:00", new Vector2f(-0.036f, 0f))
    at("6:07", new Vector2f(0.35f, -1f))
    at("18:52", new Vector2f(-0.4731, 1f))
    at("19:52", new Vector2f(-0.4f, 1f))
    
    // 6:07 - 19:52 = 6:07 + 4:08 = 10:15
    // 6:07 = 367
    // 10:15 = 615
    // 367/615 = 0.51544715447154471544715447154472
    // 0.35 - -0.4 = 0.75 * 0.515 = .386 .... 0.35 - 0.036
    // 60/615 * 0.75 = 0.0731 per hour
}

// Star color based on time of day... fully transparent
// during daylight hours
addSchedule("24:00").with {
    interpolate { t, start, end ->
        def val = start + (end - start) * t;
        env.setStarColor(val);
    }
    if( false ) {
    at("0:00", new ColorRGBA(1f, 1f, 1f, 1f));
    at("5:30", new ColorRGBA(1f, 1f, 1f, 1f));
    at("6:00", new ColorRGBA(1f, 1f, 1f, 0.25f));
    at("6:30", new ColorRGBA(1f, 1f, 1f, 0f));
    at("12:00", new ColorRGBA(1f, 1f, 1f, 0f)); 
    at("17:30", new ColorRGBA(1f, 1f, 1f, 0f));
    at("18:00", new ColorRGBA(1f, 1f, 1f, 0.5f));
    at("18:30", new ColorRGBA(1f, 1f, 1f, 1f));
    } else {
    // Adapted to a 6:08 - 19:52 day.  (+08 and +82)   
    at("0:00", new ColorRGBA(1f, 1f, 1f, 1f));
    at("5:38", new ColorRGBA(1f, 1f, 1f, 1f));
    at("6:08", new ColorRGBA(1f, 1f, 1f, 0.25f));
    at("6:38", new ColorRGBA(1f, 1f, 1f, 0f));
    at("12:00", new ColorRGBA(1f, 1f, 1f, 0f)); 
    at("19:22", new ColorRGBA(1f, 1f, 1f, 0f));
    at("19:52", new ColorRGBA(1f, 1f, 1f, 0.5f));
    at("20:22", new ColorRGBA(1f, 1f, 1f, 1f));
    }
}

// Sets the base color of the sun texture based on time of day.
addSchedule("24:00").with {
    interpolate { t, start, end ->
        def val = start + (end - start) * t;
        env.setSunColor(val);
    }
    // Old mythruna hard-coded this to: new ColorRGBA(1f,1f,1f,0.5f)
    at("0:00", new ColorRGBA(1f, 1f, 1f, 0f));
    at("5:30", new ColorRGBA(1.2f, 1.2f, 0.9f, 1.0f));
    at("6:30", new ColorRGBA(1.5f, 1.2f, 1.2f, 1.0f));
    at("7:30", new ColorRGBA(1.5f, 1.5f, 0.95f, 1.0f));
    at("16:30", new ColorRGBA(1.5f, 1.5f, 0.95f, 1.0f));
    at("17:30", new ColorRGBA(1.5f, 1.2f, 1.2f, 1.0f));
    at("18:30", new ColorRGBA(1.2f, 1.2f, 0.9f, 1.0f));    
}

// Sets the base color of the sun halo texture based on time of day.
addSchedule("24:00").with {
    interpolate { t, start, end ->
        def val = start + (end - start) * t;
        env.setSunHaloColor(val);
    }
    if( false ) {
    at("0:00", new ColorRGBA(1f, 1f, 1f, 0f));
    at("5:30", new ColorRGBA(1.2f, 1.2f, 0.2f, 0.5f));
    at("6:00", new ColorRGBA(1.2f, 1.2f, 0.7f, 0.5f));
    at("6:30", new ColorRGBA(0.8f, 0.7f, 0.3f, 0.5f));
    at("7:00", new ColorRGBA(0.6f, 0.4f, 0.1f, 0.95f));
    at("7:30", new ColorRGBA(0.5f, 0.4f, 0.3f, 0.95f));
    at("8:30", new ColorRGBA(0.8f, 0.7f, 0.7f, 0.85f));
    at("12:00", new ColorRGBA(1.2f, 1.2f, 1.2f, 0.75f));
    at("17:30", new ColorRGBA(0.8f, 0.7f, 0.3f, 0.5f));
    at("18:00", new ColorRGBA(1.2f, 1.2f, 0.7f, 0.5f));
    
    } else {
    // Original Mythruna1 schedule
    // Based on 6:08 - 19:52 day.
    at("0:00", new ColorRGBA(0.95f,0.9f, 0.7f, 0.2f));
    at("4:00", new ColorRGBA(0.95f,0.9f, 0.7f, 0.2f));
    at("5:00", new ColorRGBA(0.95f,0.9f, 0.7f, 0.5f));
    at("6:00", new ColorRGBA(0.95f,0.9f, 0.7f, 0.5f));
    //at("7:00", new ColorRGBA(0.95f,0.9f, 0.8f, 0.7f));
    at("6:30", new ColorRGBA(0.95f, 0.75f, 0.7f, 0.7f));
    at("7:00", new ColorRGBA(0.8f, 0.7f, 0.6f, 0.9f));
    at("8:00", new ColorRGBA(0.95f,0.9f, 0.6f, 0.7f));
    at("9:00", new ColorRGBA(0.95f,0.9f, 0.6f, 0.7f));
    at("10:00", new ColorRGBA(0.95f,0.9f, 0.7f, 0.8f));
    at("14:00", new ColorRGBA(0.95f,0.9f, 0.7f, 0.8f));
    at("15:00", new ColorRGBA(0.95f,0.9f, 0.6f, 0.7f));
    at("16:00", new ColorRGBA(0.95f,0.9f, 0.5f, 0.65f));
    at("17:00", new ColorRGBA(0.95f,0.9f, 0.5f, 0.65f));
    at("18:00", new ColorRGBA(0.95f,0.85f, 0.55f, 0.65f));
    at("19:00", new ColorRGBA(0.95f,0.7f, 0.55f, 0.9f));
    at("20:00", new ColorRGBA(0.95f,0.7f, 0.55f, 0.9f));
    at("21:00", new ColorRGBA(0.95f,0.6f, 0.55f, 0.2f)); 
    at("23:00", new ColorRGBA(0.95f,0.6f, 0.55f, 0.2f)); 
    }
}

// Sets the light angle based on time of day.  Syncs
// the direction to where the sun and moon are in the sky depending
// on whether it's day time or night time.
addSchedule("24:00").with {
    interpolate { t, start, end ->
        def val = start + (end - start) * t;
        val = (int)val;
        //log.info("val:" + val + " start:" + start + "  end:" + end);
        def dir;
        if( val > 0 ) {
            // Sun is the light
            // Light comes from the direction of the sun, so backwards             
            dir = env.getSunRotation().mult(new Vector3f(-1, 0, 0));
        } else {
            // Moon is the light
            dir = env.getMoonRotation().mult(new Vector3f(1, 0, 0));
        }
        env.setLightDirection(dir); 
    }

    // Based on 6:08 - 19:52 day.
    at("0:00", 0)
    at("6:08", 1)
    at("19:52", 1)
    at("19:53", 0)
}

// Sets the directional light color based on time of day 
addSchedule("24:00").with {
    interpolate { t, start, end ->
        def val = start + (end - start) * t;
        env.setDirectionalLightColor(val);
    }
    if( false ) {
    at("0:00", new ColorRGBA(0.05f, 0.05f, 0.1f, 1f));
    at("5:50", new ColorRGBA(0.05f, 0.05f, 0.1f, 1f));
    at("6:00", new ColorRGBA(0.0f, 0.0f, 0.0f, 1f));
    at("6:30", new ColorRGBA(1.5f, 1.2f, 1.1f, 1f));
    at("8:00", new ColorRGBA(1.5f, 1.5f, 1.2f, 1f));
    at("12:00", new ColorRGBA(1.5f, 1.5f, 1.1f, 1f));
    at("16:00", new ColorRGBA(1.5f, 1.5f, 1.2f, 1f));
    at("17:30", new ColorRGBA(1.5f, 1.2f, 1.1f, 1f));
    at("18:00", new ColorRGBA(0.0f, 0.0f, 0.0f, 1f));
    at("18:10", new ColorRGBA(0.05f, 0.05f, 0.1f, 1f));
    at("19:00", new ColorRGBA(0.05f, 0.05f, 0.1f, 1f));
    
    } else {
    // Original Mythruna1 schedule
    // Based on 6:08 - 19:52 day.
    at("0:00", moonDirectColor);
    //at("4:00", new ColorRGBA(0.1f, 0.1f, 0.2f, 1f));
    //at("5:00", new ColorRGBA(0.6f, 0.5f, 0.4f, 1f));
    //at("6:00", new ColorRGBA(1.2f, 0.9f, 0.8f, 1f));
    // Adjusted to new lighting scheme for day-night transition
    at("5:30", moonDirectColor); // sun is barely clearing the horizon
    // Actually, it's at 6:00 even that the sun almost tips over the horizon
    at("6:08", new ColorRGBA(0.0f, 0.0f, 0.0f, 1f)); // halfway up, light direction changes
    at("6:18", new ColorRGBA(1.2f, 0.9f, 0.8f, 1f)); // sun has fully cleared horizon 
    // done adjustments
    at("7:00", new ColorRGBA(1.4f, 1.1f, 0.9f, 1f));
    at("8:00", new ColorRGBA(1.4f, 1.3f, 1.1f, 1f));
    at("9:00", new ColorRGBA(1.4f, 1.3f, 1.2f, 1f));
    at("10:00", new ColorRGBA(1.3f, 1.3f, 1.2f, 1f));
    at("11:00", new ColorRGBA(1.2f, 1.2f, 1.1f, 1f));
    at("15:00", new ColorRGBA(1.2f, 1.2f, 1.1f, 1f));
    at("16:00", new ColorRGBA(1.3f, 1.3f, 1.2f, 1f));
    at("17:00", new ColorRGBA(1.4f, 1.3f, 1.2f, 1f));
    at("18:00", new ColorRGBA(1.4f, 1.2f, 0.9f, 1f));
    at("19:00", new ColorRGBA(1.3f, 1.1f, 0.8f, 1f));
    //at("20:00", new ColorRGBA(0.6f, 0.5f, 0.4f, 1f));
    // Adjusted to new lighting scheme for day-night transition
    at("19:42", new ColorRGBA(1.2f, 0.9f, 0.8f, 1f)); // sun still fully above horizon 
    at("19:52", new ColorRGBA(0.0f, 0.0f, 0.0f, 1f)); // halfway down, light direction changes
    at("20:00", moonDirectColor); // sun is barely clearing the horizon
    at("21:00", moonDirectColor);
    }
}

// Sets the ambient light color based on time of day
addSchedule("24:00").with {
    interpolate { t, start, end ->
        def val = start + (end - start) * t;
        env.setAmbientLightColor(val);
    }
    if( false ) {
    at("0:00", new ColorRGBA(0.05f, 0.05f, 0.1f, 1f));
    at("5:30", new ColorRGBA(0.05f, 0.05f, 0.1f, 1f));
    at("6:00", new ColorRGBA(0.1f, 0.06f, 0.05f, 1f));
    at("6:30", new ColorRGBA(0.1f, 0.06f, 0.05f, 1f));
    at("8:00", new ColorRGBA(0.1f, 0.06f, 0.05f, 1f));
    at("8:00", new ColorRGBA(0.15f, 0.15f, 0.15f, 1f));
    at("12:00", new ColorRGBA(0.4f, 0.4f, 0.6f, 1f));
    at("18:00", new ColorRGBA(0.2f, 0.08f, 0.05f, 1f));
    //at("18:00", new ColorRGBA(0.3f, 0.16f, 0.10f, 1f));
    at("19:00", new ColorRGBA(0.05f, 0.05f, 0.1f, 1f));
    } else {
    // Old Mythruna set the ambient color to White and the modulated
    // this with the material color's ambient... set to DarkGray:
    // public static final ColorRGBA DarkGray = new ColorRGBA(0.2f, 0.2f, 0.2f, 1.0f);    
    at("0:00", moonAmbientColor);
    at("5:30", moonAmbientColor);
    //at("6:08", new ColorRGBA(0.1f, 0.06f, 0.05f, 1f));
    at("6:00", new ColorRGBA(0.15, 0.08f, 0.08f, 1f));
    at("6:08", new ColorRGBA(0.2f, 0.12f, 0.1f, 1f));
    at("6:38", new ColorRGBA(0.1f, 0.06f, 0.05f, 1f));
    at("8:08", new ColorRGBA(0.1f, 0.06f, 0.05f, 1f));
    at("9:00", new ColorRGBA(0.2f, 0.2f, 0.2f, 1f));
    at("12:00", new ColorRGBA(0.2f, 0.2f, 0.2f, 1f));
    at("17:52", new ColorRGBA(0.2f, 0.2f, 0.2f, 1f));
    at("19:52", new ColorRGBA(0.3f, 0.2f, 0.15f, 1f));
    at("20:32", moonAmbientColor);
    }    
}

addSchedule("24:00").with {
    interpolate { t, start, end ->
        def val = start + (end - start) * t;
        env.fogSettings.setFogColor(val);
    }
    if( false ) {
    at("0:00", new ColorRGBA(0f, 0f, 0f, 1f));    
    at("5:30", new ColorRGBA(0f, 0f, 0f, 1f));
    at("6:00", new ColorRGBA(0.1f, 0.08f, 0.08f, 1f));
    at("6:15", new ColorRGBA(0.3f, 0.2f, 0.2f, 1f));
    at("6:30", new ColorRGBA(0.6f, 0.5f, 0.5f, 1f));
    at("7:00", new ColorRGBA(0.7f, 0.6f, 0.6f, 1f));
    at("8:00", new ColorRGBA(0.75f, 0.75f, 0.9f, 1f));
    at("12:00", new ColorRGBA(0.85f, 0.85f, 1f, 1f));
    at("16:00", new ColorRGBA(0.75f, 0.75f, 0.9f, 1f));
    at("17:00", new ColorRGBA(0.7f, 0.6f, 0.6f, 1f));
    at("17:30", new ColorRGBA(0.6f, 0.5f, 0.5f, 1f));
    at("17:45", new ColorRGBA(0.3f, 0.2f, 0.2f, 1f));
    at("18:00", new ColorRGBA(0.1f, 0.08f, 0.08f, 1f));
    at("18:30", new ColorRGBA(0.1f, 0.08f, 0.08f, 1f));
    at("19:00", new ColorRGBA(0f, 0f, 0f, 1f));
    } else {
    at("0:00", new ColorRGBA(0.0f, 0.0f, 0.1f, 1f));
    at("3:00", new ColorRGBA(0.0f, 0.0f, 0.1f, 1f));
    at("4:15", new ColorRGBA(0.1f, 0.0f, 0.2f, 1f));
    at("5:00", new ColorRGBA(0.12f, 0.1f, 0.2f, 1f));
    //at("6:00", new ColorRGBA(0.3f, 0.3f, 0.4f, 1f));
    at("6:00", new ColorRGBA(0.2f, 0.15f, 0.2f, 1f));
    at("6:08", new ColorRGBA(0.45f, 0.25f, 0.35f, 1f));
    at("6:38", new ColorRGBA(0.8f, 0.6f, 0.5f, 1f));
    at("7:00", new ColorRGBA(0.85f, 0.8f, 0.7f, 1f));
    at("8:00", new ColorRGBA(0.8f, 0.75f, 0.7f, 1f));
    at("9:00", new ColorRGBA(0.9f, 0.85f, 0.8f, 1f));
    at("10:00", new ColorRGBA(0.85f, 0.85f, 0.85f, 1f));
    at("11:00", new ColorRGBA(0.9f, 0.9f, 0.9f, 1f));
    at("14:00", new ColorRGBA(0.9f, 0.9f, 0.9f, 1f));
    at("15:00", new ColorRGBA(0.85f, 0.85f, 0.85f, 1f));
    at("16:00", new ColorRGBA(0.8f, 0.8f, 0.8f, 1f));
    at("17:00", new ColorRGBA(0.8f, 0.8f, 0.8f, 1f));
    at("18:00", new ColorRGBA(0.9f, 0.9f, 0.85f, 1f));
    at("19:00", new ColorRGBA(0.9f, 0.8f, 0.7f, 1f));
    at("19:52", new ColorRGBA(0.4f, 0.2f, 0.3f, 1f));
    at("20:00", new ColorRGBA(0.2f, 0.2f, 0.4f, 1f));
    at("21:00", new ColorRGBA(0.1f, 0.1f, 0.2f, 1f));
    at("22:00", new ColorRGBA(0.0f, 0.0f, 0.1f, 1f));
    }
}

addSchedule("24:00").with {
    interpolate { t, start, end ->
        def val = start + (end - start) * t;
        env.setSkyEast(val);
    }
    //at("0:00", new Vector2f(0.6f, -0.1f));
    if( false ) {
    at("0:00", new Vector2f(1f, 0.4f));
    at("4:00", new Vector2f(1f, 0.4f));
    at("5:00", new Vector2f(0.6f, -0.2f));
    at("5:30", new Vector2f(0.5f, -0.2f));
    at("6:00", new Vector2f(0.5f, -0.2f));
    at("6:30", new Vector2f(0.4f, -0.1f));
    at("7:00", new Vector2f(0.35f, -0.1f));
    at("8:00", new Vector2f(0.25f, -0.1f));
    at("12:00", new Vector2f(0f, -0.0f));
    at("17:00", new Vector2f(0.35f, 0.1f));
    at("18:00", new Vector2f(0.6f, 0.2f));
    at("18:30", new Vector2f(1f, 0.4f));
    at("19:00", new Vector2f(1f, 0.4f));
    at("20:00", new Vector2f(1f, 0.4f));
    } else {
    // Adapted to a 6:08 - 19:52 day.  (+08 and +82)   
    at("0:00", new Vector2f(1f, 0.4f));
    at("4:00", new Vector2f(1f, 0.4f));
    at("5:08", new Vector2f(0.6f, -0.2f));
    at("5:38", new Vector2f(0.5f, -0.2f));
    at("6:08", new Vector2f(0.5f, -0.2f));
    at("6:38", new Vector2f(0.4f, -0.1f));
    at("7:08", new Vector2f(0.35f, -0.1f));
    at("8:00", new Vector2f(0.25f, -0.1f));
    at("12:00", new Vector2f(0f, -0.0f));
    at("17:00", new Vector2f(0.35f, 0.1f));
    at("19:52", new Vector2f(0.6f, 0.2f));   
    at("20:22", new Vector2f(1f, 0.4f));
    at("20:52", new Vector2f(1f, 0.4f));
    at("21:52", new Vector2f(1f, 0.4f));
    }
}

addSchedule("24:00").with {
    interpolate { t, start, end ->
        def val = start + (end - start) * t;
        env.setSkyWest(val);
    }
    if( false ) {
    at("0:00", new Vector2f(1f, 0.4f));
    at("4:00", new Vector2f(1f, 0.4f));
    at("5:00", new Vector2f(1f, 0.4f));
    at("6:00", new Vector2f(0.75f, 0.1f));
    at("7:00", new Vector2f(0.35f, 0.1f));
    at("12:00", new Vector2f(0f, -0.0f));
    at("16:00", new Vector2f(0.25f, -0.1f));
    at("17:00", new Vector2f(0.5f, -0.0f));
    at("17:30", new Vector2f(0.5f, -0.1f));
    at("18:00", new Vector2f(0.5f, -0.3f));
    at("18:30", new Vector2f(0.5f, -0.2f));
    at("19:00", new Vector2f(0.7f, -0f));
    at("19:30", new Vector2f(1f, 0.4f));
    } else {
    // Adapted to a 6:08 - 19:52 day.  (+08 and +82)
    at("0:00", new Vector2f(1f, 0.4f));
    at("4:00", new Vector2f(1f, 0.4f));
    at("5:08", new Vector2f(1f, 0.4f));
    at("6:08", new Vector2f(0.75f, 0.1f));
    at("7:08", new Vector2f(0.35f, 0.1f));
    at("12:00", new Vector2f(0f, -0.0f));
    at("17:52", new Vector2f(0.25f, -0.1f));
    at("18:52", new Vector2f(0.5f, -0.0f));
    at("19:22", new Vector2f(0.5f, -0.1f));
    at("19:52", new Vector2f(0.5f, -0.3f));
    at("20:22", new Vector2f(0.5f, -0.2f));
    at("20:52", new Vector2f(0.7f, -0f));
    at("21:22", new Vector2f(1f, 0.4f));
    }   
}

addSchedule("24:00").with {
    interpolate { t, start, end ->
        def val = start + (end - start) * t;
        env.setSkyNorthSouth(val);
        //log.info("east:" + env.getSkyEast() + " west:" + env.getSkyWest() + " ns:" + env.getSkyNorthSouth());
    }
    if( false ) {
    //at("0:00", new Vector2f(0.75f, 0.25f));    
    at("0:00", new Vector2f(1f, 0.4f));
    at("4:00", new Vector2f(1f, 0.4f));
    at("5:00", new Vector2f(0.75f, 0.25f));
    at("5:30", new Vector2f(0.5f, 0.25f));
    at("6:00", new Vector2f(0.5f, 0.1f));
    at("6:30", new Vector2f(0.5f, 0.1f));
    at("7:00", new Vector2f(0.35f, 0.1f));
    at("12:00", new Vector2f(0f, -0.0f));
    at("17:00", new Vector2f(0.4f, 0.1f));
    at("17:30", new Vector2f(0.5f, 0.05f));
    at("18:00", new Vector2f(0.5f, 0.1f));
    at("18:30", new Vector2f(0.5f, 0.25f));
    at("19:00", new Vector2f(0.75f, 0.2f));
    at("19:30", new Vector2f(1f, 0.4f));
    } else {    
    // Adapted to a 6:08 - 19:52 day.  (+08 and +82)
    at("0:00", new Vector2f(1f, 0.4f));
    at("4:00", new Vector2f(1f, 0.4f));
    at("5:08", new Vector2f(0.75f, 0.25f));
    at("5:38", new Vector2f(0.5f, 0.25f));
    at("6:08", new Vector2f(0.5f, 0.1f));
    at("6:38", new Vector2f(0.5f, 0.1f));
    at("7:08", new Vector2f(0.35f, 0.1f));
    at("12:00", new Vector2f(0f, -0.0f));
    at("18:52", new Vector2f(0.4f, 0.1f));
    at("19:22", new Vector2f(0.5f, 0.05f));
    at("19:52", new Vector2f(0.5f, 0.1f));
    at("20:22", new Vector2f(0.5f, 0.25f));
    at("20:52", new Vector2f(0.75f, 0.2f));
    at("21:22", new Vector2f(1f, 0.4f));
    }
}

import mythruna.client.view.Bloom;

addSchedule("24:00").with {
    interpolate { t, start, end ->
        def val = start + (end - start) * t;
        env.setBloom(val);
    }        
 
    // Copied directly from old-Mythruna
    at("0:00", new Bloom(blurScale:2.7f, exposurePower:4.57f, exposureCutOff:0.189f, bloomIntensity:1.75f))
    at("6:00", new Bloom(blurScale:2.7f, exposurePower:4.57f, exposureCutOff:0.189f, bloomIntensity:1.75f))
    at("7:00", new Bloom(blurScale:0.5f, exposurePower:3.3f, exposureCutOff:0.2f, bloomIntensity:1f))
    at("8:00", new Bloom(blurScale:0.5f, exposurePower:3.3f, exposureCutOff:0.2f, bloomIntensity:1f))
    at("9:00", new Bloom(blurScale:1.0f, exposurePower:3.3f, exposureCutOff:0.2f, bloomIntensity:0.75f))
    at("10:00", new Bloom(blurScale:1.0f, exposurePower:3.3f, exposureCutOff:0.2f, bloomIntensity:0.5f))
    at("14:00", new Bloom(blurScale:1.0f, exposurePower:3.3f, exposureCutOff:0.2f, bloomIntensity:0.5f))
    at("15:00", new Bloom(blurScale:1.0f, exposurePower:3.3f, exposureCutOff:0.2f, bloomIntensity:0.75f))
    at("16:00", new Bloom(blurScale:0.5f, exposurePower:3.3f, exposureCutOff:0.2f, bloomIntensity:1f))
    at("18:00", new Bloom(blurScale:0.5f, exposurePower:3.3f, exposureCutOff:0.2f, bloomIntensity:1f))
    at("19:00", new Bloom(blurScale:2.7f, exposurePower:4.57f, exposureCutOff:0.189f, bloomIntensity:1.75f))
}


def overrides = []

//def caveBloom = new Bloom(blurScale:2.7f, exposurePower:4.57f, exposureCutOff:0.189f, bloomIntensity:1.75f);
// From original Mythruna: new ColorRGBA( 2.3f, 4.57f, 0.189f, 3.2f );
def caveBloom = new Bloom(blurScale:2.3f, exposurePower:4.57f, exposureCutOff:0.189f, bloomIntensity:3.2f);


overrides += env.addLocalOverride { Double t ->
    //log.info("t:" + t);
    def start = env.getBloom(); 
    def val = start + (caveBloom - start) * t;
    env.setBloom(val); 
}



mod.onTerminate {
    overrides.each {
        env.removeLocalOverride(it);
    }
}

